import 'dart:convert';
import 'package:flutter/foundation.dart';
import 'package:http/http.dart' as http;
import 'package:pulse/Models/commonResponse.dart';
import 'package:pulse/Models/profileResponse.dart';
import 'package:pulse/Models/webErpResponse.dart';

class ApiService {
  static const String baseUrl = "https://pulse.webgrid.in/app/";
  static const String loginUrl = "${baseUrl}Authentication/login";
  static const String logoutUrl = "${baseUrl}Home/logout";
  static const String profileUrl = "${baseUrl}Home/profile_details";
  static const String webErpUrl = "https://pulse.webgrid.in/app/Home/web_erp";

  /// Reusable POST wrapper
  static Future<http.Response?> post(
      Map<String, String> data,
      String url, [
        Map<String, String> headers = const {},
      ]) async {
    try {
      final response = await http.post(
        Uri.parse(url),
        body: data,
        headers: headers,
      );

      if (response.statusCode == 200) {
        return response;
      } else {
        debugPrint("❌ Failed POST: $url → ${response.statusCode}");
        return null;
      }
    } catch (e) {
      debugPrint("❌ Exception in POST($url): $e");
      return null;
    }
  }

  /// Login API
  static Future<CommonResponse?> login({
    required String csrfToken,
    required String email,
    required String password,
    String isApp = "1",
  }) async {
    try {
      Map<String, String> data = {
        "csrf_token_name": csrfToken,
        "email": email,
        "password": password,
        "is_app": isApp,
      };

      final res = await post(data, loginUrl);

      if (res != null) {
        return CommonResponse.fromJson(jsonDecode(res.body));
      } else {
        debugPrint("⚠️ Null Response in login()");
        return null;
      }
    } catch (e) {
      debugPrint("❌ API Error (login): $e");
      return null;
    }
  }

  /// Logout API
  static Future<CommonResponse?> logoutApi({
    required String csrfToken,
    required String sessionId,
    required String staffId,
    String isApp = "1",
  }) async {
    try {
      Map<String, String> data = {
        "csrf_token_name": csrfToken,
        "session_id": sessionId,
        "staff_id": staffId,
        "is_app": isApp,
      };

      final res = await post(data, logoutUrl);

      if (res != null) {
        return CommonResponse.fromJson(jsonDecode(res.body));
      } else {
        debugPrint("⚠️ Null Response in logoutApi()");
        return null;
      }
    } catch (e) {
      debugPrint("❌ API Error (logout): $e");
      return null;
    }
  }

  /// Profile API
  static Future<ProfileResponse?> fetchProfile({
    required String csrfToken,
    required String sessionId,
    required String staffId,
    required token,
    String isApp = "1",
  }) async {
    try {
      debugPrint("⚠️ FCM Device token: $token");
      Map<String, String> data = {
        "csrf_token_name": csrfToken,
        "session_id": sessionId,
        "staff_id": staffId,
        'token_id': (token).toString(),
        "is_app": isApp,
      };

      final res = await post(data, profileUrl);

      if (res != null) {
        return ProfileResponse.fromJson(jsonDecode(res.body));
      } else {
        debugPrint("⚠️ Null Response in profile()");
        return null;
      }
    } catch (e) {
      debugPrint("❌ API Error (profile): $e");
      return null;
    }
  }



  static Future<WebErpResponse?> fetchWebErpApi({
    required String csrfToken,
    required String sessionId,
    required String staffId,
    String isApp = "1",
  }) async {
    debugPrint("=== Web API Called ===");

    try {
      Map<String, String> data = {
        "csrf_token_name": csrfToken,
        "session_id": sessionId,
        "staff_id": staffId,
        "is_app": isApp,
      };

      debugPrint("URL: $webErpUrl");
      debugPrint("Data: $data");

      var client = http.Client();

      final response = await client.post(
        Uri.parse(webErpUrl),
        body: data,
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
          'User-Agent': 'Mozilla/5.0',
        },
      ).timeout(Duration(seconds: 30));

      debugPrint("Initial Status: ${response.statusCode}");

      http.Response finalResponse = response;

      // Handle redirect
      if (response.statusCode == 302) {
        debugPrint("Redirect detected");
        final location = response.headers['location'];
        if (location != null) {
          debugPrint("Redirecting to: $location");

          // Follow the redirect
          finalResponse = await client.get(
            Uri.parse(location),
            headers: {
              'User-Agent': 'Mozilla/5.0',
            },
          );

          debugPrint("Redirect Status: ${finalResponse.statusCode}");
          debugPrint("HTML Response Length: ${finalResponse.body.length}");
        }
      }

      client.close();

      // Return the HTML response as-is, no JSON decoding
      return WebErpResponse(html: finalResponse.body);

    } catch (e) {
      debugPrint("Error in fetchWebErpApi: $e");
      return WebErpResponse(html: "Error: $e");
    }
  }
}
