import 'package:flutter/foundation.dart';
import 'package:pulse/Models/webErpResponse.dart';
import 'package:pulse/Services/api_calling.dart';

class WebErpProvider extends ChangeNotifier {
  WebErpResponse? _response;
  bool _isLoading = false;
  String? _errorMessage;

  WebErpResponse? get response => _response;
  bool get isLoading => _isLoading;
  String? get errorMessage => _errorMessage;

  // Helper getters for common checks
  bool get hasHtmlContent => _response?.html != null;
  String? get htmlContent => _response?.html;
  bool get isSessionValid => _response?.isSessionValid ?? false;

  /// Call API and update state
  Future<void> fetchWebViewErp({
    required String csrfToken,
    required String sessionId,
    required String staffId,
    String isApp = "1",
  }) async {
    _isLoading = true;
    _errorMessage = null;
    notifyListeners();

    try {
      final res = await ApiService.fetchWebErpApi(
        csrfToken: csrfToken,
        sessionId: sessionId,
        staffId: staffId,
        isApp: isApp,
      );

      if (res != null) {
        _response = res;
        debugPrint("✅ HTML response received: ${res.html?.length ?? 0} characters");

        // Log what type of page we received
        if (res.html != null) {
          if (res.html!.toLowerCase().contains('login')) {
            debugPrint("🔐 Login page detected");
          } else if (res.html!.toLowerCase().contains('dashboard')) {
            debugPrint("📊 Dashboard page detected");
          } else if (res.html!.toLowerCase().contains('admin')) {
            debugPrint("⚙️ Admin page detected");
          }
        }
      } else {
        _errorMessage = "No response from server";
        debugPrint("❌ Null response from API");
      }
    } catch (e) {
      _errorMessage = "Error: $e";
      debugPrint("❌ API Error: $e");
    } finally {
      _isLoading = false;
      notifyListeners();
    }
  }

  /// Clear the response and error
  void clear() {
    _response = null;
    _errorMessage = null;
    notifyListeners();
  }

  /// Check if HTML contains specific text
  bool htmlContains(String text) {
    return _response?.html?.toLowerCase().contains(text.toLowerCase()) ?? false;
  }
}