import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:pulse/Models/commonResponse.dart';
import '../Services/api_calling.dart';

class AuthProvider extends ChangeNotifier {
  bool _isLoading = false;
  CommonResponse? _loginResponse;
  String? _errorMessage;
  bool _isLoggedIn = false;

  // Existing Getters
  bool get isLoading => _isLoading;
  CommonResponse? get loginResponse => _loginResponse;
  String? get errorMessage => _errorMessage;
  bool get isLoggedIn => _isLoggedIn;

  // New Getters for convenience
  String? get sessionId => _loginResponse?.sessionId;
  String? get staffId => _loginResponse?.staffId;
  String? get message => _loginResponse?.message;
  String? get error => _loginResponse?.error;

  /// Login API call
  Future<void> loginUser(
      BuildContext context, {
        required String csrfToken,
        required String email,
        required String password,
      }) async {
    try {
      _isLoading = true;
      _errorMessage = null;
      _loginResponse = null;
      notifyListeners();

      final data = await ApiService.login(
        csrfToken: csrfToken,
        email: email,
        password: password,
      );

      if (data != null) {
        _loginResponse = data;

        if (_loginResponse?.error == "0") {
          _isLoggedIn = true;
          debugPrint("✅ Login successful: ${_loginResponse?.toJson()}");
        } else {
          _isLoggedIn = false;
          _errorMessage = _loginResponse?.message ?? "Login failed";
          debugPrint("⚠️ Login failed: $_errorMessage");
        }
      } else {
        _errorMessage = "No response from server";
        debugPrint("⚠️ $_errorMessage");
      }
    } catch (e) {
      _isLoggedIn = false;
      _errorMessage = "❌ Exception: $e";
      debugPrint(_errorMessage);
    } finally {
      _isLoading = false;
      notifyListeners();
    }
  }

  /// Logout API call
  Future<void> logoutUser({
    required String csrfToken,
    required String sessionId,
    required String staffId,
  }) async {
    try {
      _isLoading = true;
      _errorMessage = null;
      notifyListeners();

      final res = await ApiService.logoutApi(
        csrfToken: csrfToken,
        sessionId: sessionId,
        staffId: staffId,
      );

      if (res != null) {
        if (res.error == "0") {
          _isLoggedIn = false;
          _loginResponse = null; // clear user session
          debugPrint("✅ Logout successful: ${res.toJson()}");
        } else {
          _errorMessage = res.message ?? "Logout failed";
          debugPrint("⚠️ Logout failed: $_errorMessage");
        }
      } else {
        _errorMessage = "No response from server";
        debugPrint("⚠️ $_errorMessage");
      }
    } catch (e) {
      _errorMessage = "❌ Exception: $e";
      debugPrint(_errorMessage);
    } finally {
      _isLoading = false;
      notifyListeners();
    }
  }

  /// Reset state if needed
  void reset() {
    _isLoading = false;
    _loginResponse = null;
    _errorMessage = null;
    _isLoggedIn = false;
    notifyListeners();
  }
}
