import 'package:flutter/material.dart';
import 'package:provider/provider.dart';
import '../../Notifiers/TransactionsProvider.dart';
import '../../Utility/AppColors.dart';

class PaymentdetailDialog extends StatefulWidget {
  final String sessionId;
  final String accId;
  final String billId;

  const PaymentdetailDialog({
    super.key,
    required this.sessionId,
    required this.accId,
    required this.billId,
  });

  @override
  State<PaymentdetailDialog> createState() => _PaymentdetailDialogState();
}

class _PaymentdetailDialogState extends State<PaymentdetailDialog> {
  @override
  void initState() {
    super.initState();
    Future.microtask(() {
      final provider = Provider.of<TransactionsProvider>(context, listen: false);
      provider.fetchPaymentDetails(widget.accId, widget.sessionId, widget.billId);
    });
  }

  @override
  Widget build(BuildContext context) {
    final provider = Provider.of<TransactionsProvider>(context);
    final paymentData = provider.paymentDetail?.paymentDetails;

    return Dialog(
      insetPadding: const EdgeInsets.symmetric(horizontal: 24),
      backgroundColor: Colors.white,
      shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(20)),
      child: Container(
        width: 360,
        padding: const EdgeInsets.all(20),
        decoration: BoxDecoration(
          borderRadius: BorderRadius.circular(20),
          color: Colors.white,
        ),
        child: provider.isPaymentLoading
            ? const SizedBox(
          height: 200,
          child: Center(child: CircularProgressIndicator()),
        )
            : paymentData == null
            ? const Padding(
          padding: EdgeInsets.all(32),
          child: Center(
            child: Text(
              "No payment details found.",
              style: TextStyle(
                fontFamily: "Poppins",
                color: Colors.black54,
              ),
            ),
          ),
        )
            : Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            // ❌ Close button
            Row(
              mainAxisAlignment: MainAxisAlignment.end,
              children: [
                InkResponse(
                  onTap: () => Navigator.pop(context),
                  child: const Icon(Icons.close,
                      color: Colors.black54, size: 20),
                ),
              ],
            ),

            // ✅ Success icon
            const SizedBox(height: 4),
            Container(
              height: 64,
              width: 64,
              decoration: const BoxDecoration(
                color: Color(0xFFDFFFE5),
                shape: BoxShape.circle,
              ),
              child: const Icon(
                Icons.check_circle,
                color: Colors.green,
                size: 44,
              ),
            ),

            const SizedBox(height: 16),

            // 💰 Amount & Status
            Text(
              "₹${paymentData.amount ?? "--"}",
              style: const TextStyle(
                fontSize: 30,
                fontWeight: FontWeight.bold,
                fontFamily: "Poppins",
                color: Colors.black,
              ),
            ),
            const SizedBox(height: 6),
            const Text(
              "Bill Paid",
              style: TextStyle(
                fontSize: 16,
                fontFamily: "Poppins",
                color: Colors.black54,
              ),
            ),
            const SizedBox(height: 6),

            // 🔹 Product name / bill name (example static or from API)
            Text(
              "5KV Silent Diesel Generator",
              style: TextStyle(
                color: AppColors.amountText,
                fontFamily: "Poppins",
                fontWeight: FontWeight.w600,
                decoration: TextDecoration.underline,
              ),
            ),

            const SizedBox(height: 16),
            const Divider(thickness: 1, color: Color(0xFFE6E6E6)),
            const SizedBox(height: 8),

            // 🧾 Transaction Info Section
            const Align(
              alignment: Alignment.centerLeft,
              child: Text(
                "Transaction Info",
                style: TextStyle(
                  fontSize: 14,
                  fontWeight: FontWeight.w600,
                  fontFamily: "Poppins",
                  color: Colors.black,
                ),
              ),
            ),
            const SizedBox(height: 8),
            // _buildRow("Reference No.", paymentData.refNo ?? "--"),
            _buildRow("Payment Date", paymentData.datetime ?? "--"),
            _buildRow("Mode", paymentData.mode ?? "--"),

            const SizedBox(height: 12),
            const Divider(thickness: 1, color: Color(0xFFE6E6E6)),
            const SizedBox(height: 8),

            // 💳 Payment Details
            const Align(
              alignment: Alignment.centerLeft,
              child: Text(
                "Payment Details",
                style: TextStyle(
                  fontSize: 14,
                  fontWeight: FontWeight.w600,
                  fontFamily: "Poppins",
                  color: Colors.black,
                ),
              ),
            ),
            const SizedBox(height: 8),
            _buildRow("Total Adjusted Amount", "₹${paymentData.amount ?? "0"}"),
            _buildRow("Remaining Amount", "₹${paymentData.remainBalance ?? "0"}"),

            const SizedBox(height: 24),

            // 📥 Download Button
            SizedBox(
              width: double.infinity,
              child: ElevatedButton.icon(
                onPressed: () {
                  // TODO: connect to downloadReceipt method
                },
                icon: const Icon(Icons.download, color: Colors.white, size: 18),
                label: const Text(
                  "Download Receipt",
                  style: TextStyle(
                    fontSize: 15,
                    fontFamily: "Poppins",
                    fontWeight: FontWeight.w600,
                    color: Colors.white,
                  ),
                ),
                style: ElevatedButton.styleFrom(
                  backgroundColor: AppColors.amountText,
                  shape: RoundedRectangleBorder(
                    borderRadius: BorderRadius.circular(50),
                  ),
                  padding: const EdgeInsets.symmetric(vertical: 14),
                ),
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildRow(String title, String value) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 4),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Expanded(
            flex: 4,
            child: Text(
              title,
              maxLines: 3,
              style: const TextStyle(
                fontSize: 14,
                fontFamily: "Poppins",
                color: Colors.black54,
              ),
            ),
          ),
          Text(
            value,
            maxLines: 3,
            style: const TextStyle(
              fontSize: 14,
              fontFamily: "Poppins",
              fontWeight: FontWeight.w500,
              color: Colors.black,
            ),
          ),
        ],
      ),
    );
  }
}
