import 'dart:convert';
import 'package:flutter/foundation.dart';
import 'package:gen_service/Services/api_calling.dart';

import '../Models/CommonResponse.dart';
import '../Models/HelpAndComplaintModels/ComplaintListResponse.dart';
import '../Models/HelpAndComplaintModels/GeneratorListResponse.dart';
import '../Screens/HelpAndComplaintScreens/DropDownsListResponse.dart';

class HelpAndComplaintProvider extends ChangeNotifier {
  bool _isLoading = false;
  String? _errorMessage;
  ComplaintListResponse? _complaintListResponse;

  bool get isLoading => _isLoading;
  String? get errorMessage => _errorMessage;
  ComplaintListResponse? get complaintListResponse => _complaintListResponse;

  GeneratorListResponse? _generatorListResponse;

  GeneratorListResponse? get generatorListResponse => _generatorListResponse;

  DropDownsListResponse? _dropDownsListResponse;
  DropDownsListResponse? get dropDownsListResponse => _dropDownsListResponse;


  ///----------------------------------------------
  /// Fetch Complaints List
  ///----------------------------------------------
  Future<void> fetchComplaintsList({
    required String accId,
    required String sessionId,
  }) async {
    _setLoading(true);
    try {
      final response = await ApiCalling.fetchComplaintsListApi(accId, sessionId);
      if (response != null && response.error == "0") {
        _complaintListResponse = response;
        _errorMessage = null;
      } else {
        _errorMessage = response?.message ?? "Failed to fetch complaints";
      }
    } catch (e) {
      _errorMessage = "Error fetching complaints: $e";
      debugPrint(_errorMessage);
    } finally {
      _setLoading(false);
    }
  }

  ///----------------------------------------------
  /// Add Complaint
  ///----------------------------------------------
  Future<CommonResponse?> addComplaint({
    required String accId,
    required String sessionId,
    required String genId,
    required String categoryId,
    required String descriptionId,
    required String note,
    required String typeId,
  }) async {
    _setLoading(true);
    try {
      final response = await ApiCalling.addComplaintApi(
        accId,
        sessionId,
        genId,
        categoryId,
        descriptionId,
        note,
        typeId,
      );

      if (response != null && response.error == "0") {
        // Optionally refresh list after successful submission
        await fetchComplaintsList(accId: accId, sessionId: sessionId);
        return response;
      } else {
        _errorMessage = response?.message ?? "Failed to add complaint";
        return response;
      }
    } catch (e) {
      _errorMessage = "Error adding complaint: $e";
      debugPrint(_errorMessage);
      return null;
    } finally {
      _setLoading(false);
    }
  }


  //  NEW FUNCTION: Fetch Generator List for Help Screen
  Future<void> fetchGeneratorList({
    required String accId,
    required String sessionId,
  }) async {
    _setLoading(true);
    try {
      final response = await ApiCalling.addGeneratorListApi(accId, sessionId);
      if (response != null && response.error == "0") {
        _generatorListResponse = response;
        _errorMessage = null;
      } else {
        _errorMessage = response?.message ?? "Failed to fetch generator list";
      }
    } catch (e) {
      _errorMessage = "Error fetching generator list: $e";
      debugPrint(_errorMessage);
    } finally {
      _setLoading(false);
    }
  }


  ///----------------------------------------------
  /// Fetch Complaint Dropdowns List
  ///----------------------------------------------
  Future<void> fetchComplaintDropdowns({
    required String accId,
    required String sessionId,
    required String genId,
  }) async {
    _setLoading(true);
    try {
      final response = await ApiCalling.dropdownsListApi(accId, sessionId, genId);
      if (response != null && response.error == "0") {
        _dropDownsListResponse = response;
        _errorMessage = null;
      } else {
        _errorMessage = response?.message ?? "Failed to fetch dropdown data";
      }
    } catch (e) {
      _errorMessage = "Error fetching dropdown list: $e";
      debugPrint(_errorMessage);
    } finally {
      _setLoading(false);
    }
  }




  ///----------------------------------------------
  /// Private Helpers
  ///----------------------------------------------
  void _setLoading(bool value) {
    _isLoading = value;
    notifyListeners();
  }

  void clearData() {
    _complaintListResponse = null;
    _errorMessage = null;
    notifyListeners();
  }
}
