// utils/custom_snackbar.dart
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';

class CustomSnackBar {
  static void show({
    required BuildContext context,
    required String message,
    String? title,
    IconData? icon,
    Color backgroundColor = const Color(0xFF324563),
    Duration duration = const Duration(seconds: 4),
    SnackBarAction? action,
    bool showCloseIcon = true,
    bool enableHaptic = true,
    BorderRadiusGeometry? borderRadius,
    List<BoxShadow>? customShadow,
    Gradient? gradient,
  }) {
    if (enableHaptic) {
      HapticFeedback.lightImpact();
    }

    final theme = Theme.of(context);

    ScaffoldMessenger.of(context).showSnackBar(
      SnackBar(
        backgroundColor: Colors.transparent,
        elevation: 0,
        duration: duration,
        behavior: SnackBarBehavior.floating,
        padding: EdgeInsets.zero,
        content: Container(
          margin: const EdgeInsets.symmetric(horizontal: 16),
          padding: const EdgeInsets.all(16),
          decoration: BoxDecoration(
            color: backgroundColor,
            borderRadius: borderRadius ?? BorderRadius.circular(16),
            boxShadow: customShadow ?? [
              BoxShadow(
                color: backgroundColor.withOpacity(0.3),
                blurRadius: 20,
                offset: const Offset(0, 8),
                spreadRadius: 1,
              ),
              BoxShadow(
                color: Colors.black.withOpacity(0.1),
                blurRadius: 4,
                offset: const Offset(0, 2),
              ),
            ],
            gradient: gradient ?? LinearGradient(
              begin: Alignment.topLeft,
              end: Alignment.bottomRight,
              colors: [
                backgroundColor,
                Color.alphaBlend(Colors.white.withOpacity(0.1), backgroundColor),
              ],
            ),
            border: Border.all(
              color: Colors.white.withOpacity(0.1),
              width: 1,
            ),
          ),
          child: Row(
            children: [
              // Icon with glowing effect
              Container(
                padding: const EdgeInsets.all(10),
                decoration: BoxDecoration(
                  color: Colors.white.withOpacity(0.15),
                  shape: BoxShape.circle,
                  boxShadow: [
                    BoxShadow(
                      color: backgroundColor.withOpacity(0.5),
                      blurRadius: 10,
                      spreadRadius: 2,
                    ),
                  ],
                ),
                child: Icon(
                  icon ?? Icons.notifications_none_rounded,
                  color: Colors.white,
                  size: 22,
                ),
              ),

              const SizedBox(width: 16),

              // Content Section
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  mainAxisSize: MainAxisSize.min,
                  children: [
                    if (title != null) ...[
                      Text(
                        title,
                        style: const TextStyle(
                          color: Colors.white,
                          fontWeight: FontWeight.w700,
                          fontSize: 15,
                          letterSpacing: -0.2,
                        ),
                      ),
                      const SizedBox(height: 4),
                    ],
                    Text(
                      message,
                      style: TextStyle(
                        color: Colors.white.withOpacity(0.9),
                        fontSize: 14,
                        height: 1.3,
                        fontWeight: FontWeight.w400,
                      ),
                      maxLines: 3,
                      overflow: TextOverflow.ellipsis,
                    ),

                    // Action Button
                    if (action != null) ...[
                      const SizedBox(height: 8),
                      Align(
                        alignment: Alignment.centerLeft,
                        child: Container(
                          decoration: BoxDecoration(
                            borderRadius: BorderRadius.circular(8),
                            border: Border.all(
                              color: Colors.white.withOpacity(0.3),
                            ),
                          ),
                          child: TextButton(
                            onPressed: action.onPressed,
                            style: TextButton.styleFrom(
                              padding: const EdgeInsets.symmetric(
                                horizontal: 12,
                                vertical: 4,
                              ),
                              minimumSize: Size.zero,
                              tapTargetSize: MaterialTapTargetSize.shrinkWrap,
                            ),
                            child: Text(
                              action.label,
                              style: TextStyle(
                                color: Colors.white,
                                fontSize: 12,
                                fontWeight: FontWeight.w600,
                              ),
                            ),
                          ),
                        ),
                      ),
                    ],
                  ],
                ),
              ),

              const SizedBox(width: 8),

              // Close Button with better styling
              if (showCloseIcon)
                GestureDetector(
                  onTap: () {
                    ScaffoldMessenger.of(context).hideCurrentSnackBar();
                  },
                  child: Container(
                    padding: const EdgeInsets.all(4),
                    decoration: BoxDecoration(
                      color: Colors.white.withOpacity(0.1),
                      shape: BoxShape.circle,
                    ),
                    child: Icon(
                      Icons.close_rounded,
                      color: Colors.white.withOpacity(0.7),
                      size: 18,
                    ),
                  ),
                ),
            ],
          ),
        ),
      ),
    );
  }

  // Premium Success Snackbar
  static void showSuccess({
    required BuildContext context,
    required String message,
    String title = "Success",
    Color backgroundColor = const Color(0xFF059669),
    SnackBarAction? action,
    bool enableHaptic = true,
  }) {
    show(
      context: context,
      message: message,
      title: title,
      icon: Icons.check_circle_rounded,
      backgroundColor: backgroundColor,
      gradient: const LinearGradient(
        begin: Alignment.topLeft,
        end: Alignment.bottomRight,
        colors: [
          Color(0xFF2E7D32),
          Color(0xFF4CAF50),
        ],
      ),
      action: action,
      enableHaptic: enableHaptic,
    );
  }

  // Premium Error Snackbar
  static void showError({
    required BuildContext context,
    required String message,
    String title = "Error",
    Color backgroundColor = const Color(0xFFDC2626),
    SnackBarAction? action,
    bool enableHaptic = true,
  }) {
    show(
      context: context,
      message: message,
      title: title,
      icon: Icons.error_outline_rounded,
      backgroundColor: backgroundColor,
      gradient: const LinearGradient(
        begin: Alignment.topLeft,
        end: Alignment.bottomRight,
        colors: [
          Color(0xFFB22222),
          Color(0xFFB22222),
        ],
      ),
      action: action,
      enableHaptic: enableHaptic,
    );
  }

  // Premium Warning Snackbar
  static void showWarning({
    required BuildContext context,
    required String message,
    String title = "Warning",
    Color backgroundColor = const Color(0xFFFFA000),
    SnackBarAction? action,
    bool enableHaptic = true,
  }) {
    show(
      context: context,
      message: message,
      title: title,
      icon: Icons.warning_amber_rounded,
      backgroundColor: backgroundColor,
      gradient: const LinearGradient(
        begin: Alignment.topLeft,
        end: Alignment.bottomRight,
        colors: [
          Color(0xFFF57F17),
          Color(0xFFF59E0B),
        ],
      ),
      action: action,
      enableHaptic: enableHaptic,
    );
  }

  // Premium Info Snackbar
  static void showInfo({
    required BuildContext context,
    required String message,
    String title = "Info",
    SnackBarAction? action,
    bool enableHaptic = true,
  }) {
    show(
      context: context,
      message: message,
      title: title,
      icon: Icons.info_outline_rounded,
      backgroundColor: const Color(0xFF2563EB),
      gradient: const LinearGradient(
        begin: Alignment.topLeft,
        end: Alignment.bottomRight,
        colors: [
          Color(0xFF2563EB),
          Color(0xFF0D47A1),
        ],
      ),
      action: action,
      enableHaptic: enableHaptic,
    );
  }

  // Premium Snackbar with custom color
  static void showCustom({
    required BuildContext context,
    required String message,
    required String title,
    required Color backgroundColor,
    required IconData icon,
    SnackBarAction? action,
    bool enableHaptic = true,
    Gradient? gradient,
  }) {
    show(
      context: context,
      message: message,
      title: title,
      icon: icon,
      backgroundColor: backgroundColor,
      gradient: gradient,
      action: action,
      enableHaptic: enableHaptic,
    );
  }
}