import 'package:flutter/material.dart';
import 'package:flutter_svg/flutter_svg.dart';
import 'package:gen_rentals/Screens/HelpScreens/CreateTicketScreen.dart';
import 'package:gen_rentals/Screens/HelpScreens/ProcessTicketScreen.dart';
import 'package:gen_rentals/Utility/Reusablewidgets.dart';

import '../../Utility/AppColors.dart';

class HelpScreen extends StatefulWidget {
  const HelpScreen({super.key});

  @override
  State<HelpScreen> createState() => _HelpScreenState();
}

class _HelpScreenState extends State<HelpScreen> {
  // Dummy data for help - with proper null safety
  final List<Map<String, dynamic>> createNewTickets = [
    {
      'title': 'Payment Issues',
      'description': 'Get help with payment related problems',
      'icon': "assets/svg/rupee_coin_ic.svg",
      'color': Color(0xFFFFEFBE),
    },
    {
      'title': 'Bill Related Issues',
      'description': 'Resolve bill and invoice matters',
      'icon': "assets/svg/know_pay.svg",
      'color': Color(0xFFCEF9FF),
    },
    {
      'title': 'Other Issues',
      'description': 'Any other support you need',
      'icon': 'assets/svg/help_ic.svg',
      'color': Color(0xFFE4E5FF),
    },
  ];

  final List<Map<String, String>> processingTickets = [
    {'title': 'Payment Issue', 'date': '25th Jan 2025', 'status': 'In Process'},
  ];

  final List<Map<String, String>> closedTickets = [
    {'title': 'Bill Payments', 'date': '25th Jan 2025'},
    {'title': 'Others', 'date': '25th Jan 2025'},
    {'title': 'Payment Issue', 'date': '25th Jan 2025'},
  ];

  @override
  Widget build(BuildContext context) {
    return SafeArea(
      top: false,
      child: Scaffold(
        backgroundColor: AppColors.backgroundRegular,
        appBar: AppBar(
          automaticallyImplyLeading: false,
          backgroundColor: Colors.white,
          title: Row(
            children: [
              InkResponse(
                onTap: () => Navigator.pop(context, true),
                child: SvgPicture.asset(
                  "assets/svg/continue_left_ic.svg",
                  height: 25,
                ),
              ),
              const SizedBox(width: 10),
              Text(
                "Help?",
                style: TextStyle(
                  fontSize: 16,
                  fontFamily: "Plus Jakarta Sans",
                  fontWeight: FontWeight.w600,
                  color: Colors.black87,
                ),
              ),
            ],
          ),
        ),
        // Main content
        body: SingleChildScrollView(
          padding: const EdgeInsets.all(16),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              // Create New Ticket Section
              SectionHeading(
                title: 'Create New Ticket',
                padding: EdgeInsets.symmetric(horizontal: 2, vertical: 4),
              ),
              const SizedBox(height: 12),
              _buildCreateNewTicketSection(),
              const SizedBox(height: 12),

              // Processing Tickets Section
              SectionHeading(
                title: 'Processing Tickets',
                padding: EdgeInsets.symmetric(horizontal: 2, vertical: 4),
              ),
              const SizedBox(height: 2),
              _buildProcessingTicketsSection(),
              const SizedBox(height: 10),

              // Closed Tickets Section
              SectionHeading(
                title: 'Closed Tickets',
                padding: EdgeInsets.symmetric(horizontal: 2, vertical: 4),
              ),
              const SizedBox(height: 2),
              _buildClosedTicketsSection(),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildCreateNewTicketSection() {
    return GridView.builder(
      shrinkWrap: true,
      physics: const NeverScrollableScrollPhysics(),
      gridDelegate: const SliverGridDelegateWithFixedCrossAxisCount(
        crossAxisCount: 3,
        crossAxisSpacing: 12,
        mainAxisSpacing: 12,
        childAspectRatio: 0.99,
      ),
      itemCount: createNewTickets.length,
      itemBuilder: (context, index) {
        final ticket = createNewTickets[index];
        final String title = ticket['title'] ?? 'Unknown';
        final String description = ticket['description'] ?? '';
        final String icon = ticket['icon'] ?? 'assets/svg/help_ic.svg';
        final Color color = ticket['color'] ?? Colors.grey;

        return _buildFeatureCard(
          title: title,
          description: description,
          icon: icon,
          color: color,
        );
      },
    );
  }

  Widget _buildFeatureCard({
    required String title,
    required String description,
    required String icon,
    required Color color,
  }) {
    return GestureDetector(
      onTap: () {
        Navigator.push(
            context,
            MaterialPageRoute(builder: (context) => HelpTicketScreen(reason: title,))
        );
      },
      child: Container(
        padding: const EdgeInsets.symmetric(horizontal: 2, vertical: 1),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.center,
          children: [
            // Icon container
            Container(
              width: 88,
              height: 88,
              decoration: BoxDecoration(
                color: color.withOpacity(0.7),
                borderRadius: BorderRadius.circular(12),
              ),
              child: Center(
                child: SizedBox(
                  height: 40,
                  width: 40,
                  child: SvgPicture.asset(
                    icon,
                    fit: BoxFit.fitWidth,
                  ),
                ),
              ),
            ),
            const SizedBox(height: 8),
            // Title
            SizedBox(
              child: Text(
                title,
                textAlign: TextAlign.center,
                style: TextStyle(
                  color: AppColors.nearDarkText,
                  fontSize: 14,
                  fontWeight: FontWeight.w400,
                  fontFamily: "Plus Jakarta Sans",
                ),
              ),
            ),
            const SizedBox(height: 4),
          ],
        ),
      ),
    );
  }

  Widget _buildProcessingTicketsSection() {
    return Container(
      padding: const EdgeInsets.all(4),
      child: Column(
        children: processingTickets.map((ticket) {
          return Padding(
            padding: const EdgeInsets.only(bottom: 4),
            child: CommonListItem(
              title: ticket['title']!,
              date: ticket['date']!,
              status: ticket['status']!,
              onTap: () {
                Navigator.push(
                    context,
                    MaterialPageRoute(builder: (context) => ProcessTicketChatScreen())
                );
              },
            ),
          );
        }).toList(),
      ),
    );
  }

  Widget _buildClosedTicketsSection() {
    return Container(
      padding: const EdgeInsets.all(4),
      child: Column(
        children: closedTickets.map((ticket) {
          return Padding(
            padding: const EdgeInsets.only(bottom: 4),
            child: CommonListItem(
              title: ticket['title']!,
              date: ticket['date']!,
              status: "", // Empty status for closed tickets
              onTap: () {
                // Handle closed ticket tap
              },
            ),
          );
        }).toList(),
      ),
    );
  }
}

class CommonListItem extends StatelessWidget {
  final String title;
  final String date;
  final String status;
  final VoidCallback? onTap;

  const CommonListItem({
    Key? key,
    required this.title,
    required this.date,
    required this.status,
    this.onTap,
  }) : super(key: key);

  @override
  Widget build(BuildContext context) {
    return Column(
      children: [
        const SizedBox(height: 4),
        Material(
          color: Colors.transparent,
          child: InkWell(
            borderRadius: BorderRadius.circular(12),
            onTap: onTap,
            child: Container(
              decoration: BoxDecoration(
                color: Colors.white,
                borderRadius: BorderRadius.circular(12),
              ),
              padding: const EdgeInsets.symmetric(vertical: 14, horizontal: 16),
              child: Row(
                mainAxisAlignment: MainAxisAlignment.spaceBetween,
                children: [
                  Expanded(
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Text(
                          title,
                          style: const TextStyle(
                            fontSize: 14,
                            fontFamily: "Plus Jakarta Sans",
                            fontWeight: FontWeight.w500,
                            color: Colors.black87,
                          ),
                        ),
                        const SizedBox(height: 4),
                        Text(
                          date,
                          style: TextStyle(
                            fontSize: 12,
                            fontFamily: "Plus Jakarta Sans",
                            fontWeight: FontWeight.w400,
                            color: Colors.grey[600],
                          ),
                        ),
                      ],
                    ),
                  ),
                  if (status.isNotEmpty)
                    Container(
                      padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 6),
                      decoration: BoxDecoration(
                        color: Colors.green.withOpacity(0.2),
                        borderRadius: BorderRadius.circular(6),
                      ),
                      child: Text(
                        status,
                        style: const TextStyle(
                          fontSize: 12,
                          fontFamily: "Plus Jakarta Sans",
                          fontWeight: FontWeight.w500,
                          color: Colors.black87,
                        ),
                      ),
                    ),
                ],
              ),
            ),
          ),
        ),
      ],
    );
  }
}
