import 'package:flutter/material.dart';
import 'package:gen_rentals/Notifier/TransactionsProvider.dart';
import 'package:provider/provider.dart';
import '../../Utility/AppColors.dart';
import '../../Utility/Reusablewidgets.dart'; // for CustomSnackBar if needed

class BillStatusToast extends StatefulWidget {
  final String sessionId;
  final String accId;
  final String ledgerId;

  const BillStatusToast({
    super.key,
    required this.sessionId,
    required this.accId,
    required this.ledgerId,
  });

  @override
  State<BillStatusToast> createState() => _BillStatusToastState();
}

class _BillStatusToastState extends State<BillStatusToast> {
  @override
  void initState() {
    super.initState();

    // ✅ Automatically fetch data when dialog opens
    Future.microtask(() {
      final provider =
      Provider.of<TransactionsProvider>(context, listen: false);
      provider.fetchPaymentReceiptDetails(
        widget.sessionId,
        widget.accId,
        widget.ledgerId,
      );
    });
  }

  @override
  Widget build(BuildContext context) {
    final provider = Provider.of<TransactionsProvider>(context);
    final data = provider.receiptDetails;

    return Dialog(
      insetPadding: const EdgeInsets.symmetric(horizontal: 24),
      backgroundColor: Colors.white,
      shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(16)),
      child: Container(
        width: 360,
        padding: const EdgeInsets.all(20),
        decoration: BoxDecoration(
          borderRadius: BorderRadius.circular(16),
          color: Colors.white,
        ),
        child: provider.isLoading
            ? const SizedBox(
            height: 200,
            child: Center(child: CircularProgressIndicator()))
            : data == null
            ? const Padding(
          padding: EdgeInsets.all(32),
          child: Center(
            child: Text(
              "No receipt details found.",
              style: TextStyle(
                fontFamily: "Poppins",
                color: Colors.black54,
              ),
            ),
          ),
        )
            : Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            // ✅ Success Tick
            Container(
              height: 60,
              width: 60,
              decoration: const BoxDecoration(
                color: Color(0xFFDFFFEB),
                shape: BoxShape.circle,
              ),
              child: const Icon(
                Icons.check_circle,
                color: Colors.green,
                size: 40,
              ),
            ),
            const SizedBox(height: 16),

            // ✅ Title & Amount
            const Text(
              "Payment Receipt",
              style: TextStyle(
                fontSize: 18,
                fontFamily: "Poppins",
                fontWeight: FontWeight.w600,
                color: Colors.black,
              ),
            ),
            const SizedBox(height: 8),
            Text(
              "₹${data.cAmount ?? "--"}",
              style: const TextStyle(
                fontSize: 28,
                fontWeight: FontWeight.bold,
                fontFamily: "Poppins",
                color: Colors.black,
              ),
            ),

            const SizedBox(height: 12),
            const Divider(thickness: 1, color: Color(0xFFE6E6E6)),
            const SizedBox(height: 12),

            // ✅ Payment Details
            const Align(
              alignment: Alignment.centerLeft,
              child: Text(
                "Payment Details",
                style: TextStyle(
                  fontSize: 14,
                  fontWeight: FontWeight.w600,
                  fontFamily: "Poppins",
                  color: Colors.black,
                ),
              ),
            ),
            const SizedBox(height: 8),
            _buildRow("Order ID", "#${data.id ?? "--"}", Colors.blue),
            _buildRow("Date", data.prDate ?? "--", Colors.black),
            _buildRow("Payment Mode", data.mode ?? "--",
                Colors.black),

            const SizedBox(height: 12),
            const Divider(thickness: 1, color: Color(0xFFE6E6E6)),
            const SizedBox(height: 12),

            // ✅ Products
            const Align(
              alignment: Alignment.centerLeft,
              child: Text(
                "Products",
                style: TextStyle(
                  fontSize: 14,
                  fontWeight: FontWeight.w600,
                  fontFamily: "Poppins",
                  color: Colors.black,
                ),
              ),
            ),
            const SizedBox(height: 8),
            _buildRow(
              data.narration ?? "Product Item",
              "₹${data.dAmount ?? "--"}",
              Colors.black,
            ),

            const SizedBox(height: 12),
            const Divider(thickness: 1, color: Color(0xFFE6E6E6)),
            const SizedBox(height: 8),

            // ✅ Total
            Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                const Text(
                  "Total",
                  style: TextStyle(
                    fontSize: 15,
                    fontFamily: "Poppins",
                    fontWeight: FontWeight.w600,
                    color: Colors.black,
                  ),
                ),
                Text(
                  "₹${data.cAmount ?? "--"}",
                  style: const TextStyle(
                    fontSize: 15,
                    fontFamily: "Poppins",
                    fontWeight: FontWeight.w600,
                    color: Colors.black,
                  ),
                ),
              ],
            ),

            const SizedBox(height: 22),

            // ✅ Download Button
            SizedBox(
              width: double.infinity,
              child: ElevatedButton.icon(
                onPressed: provider.isDownloading
                    ? null
                    : () async {
                  await provider.downloadPaymentReceipt(
                    context,
                    widget.sessionId,
                    widget.ledgerId,
                    widget.accId,
                  );
                },
                icon: provider.isDownloading
                    ? const SizedBox(
                  width: 18,
                  height: 18,
                  child: CircularProgressIndicator(
                    strokeWidth: 2,
                    color: Colors.white,
                  ),
                )
                    : const Icon(Icons.download,
                    color: Colors.white, size: 18),
                label: Text(
                  provider.isDownloading
                      ? "Downloading..."
                      : "Download Receipt",
                  style: const TextStyle(
                    fontSize: 15,
                    fontFamily: "Poppins",
                    fontWeight: FontWeight.w600,
                    color: Colors.white,
                  ),
                ),
                style: ElevatedButton.styleFrom(
                  backgroundColor: AppColors.amountText,
                  shape: RoundedRectangleBorder(
                    borderRadius: BorderRadius.circular(50),
                  ),
                  padding:
                  const EdgeInsets.symmetric(vertical: 14),
                ),
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildRow(String title, String value, Color valueColor) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 4),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Expanded(
            flex: 5,
            child: Text(
              title,
              style: const TextStyle(
                fontSize: 14,
                fontFamily: "Poppins",
                color: Colors.black54,
              ),
            ),
          ),
          Text(
            value,
            style: TextStyle(
              fontSize: 14,
              fontFamily: "Poppins",
              fontWeight: FontWeight.w500,
              color: valueColor,
            ),
          ),
        ],
      ),
    );
  }
}
