import 'package:flutter/material.dart';
import 'package:provider/provider.dart';
import '../../Utility/AppColors.dart';
import '../../Notifier/TransactionsProvider.dart';

class BillPendingToast extends StatefulWidget {
  final String sessionId;
  final String accId;
  final String ledgerId;
  final VoidCallback onPayNow;

  const BillPendingToast({
    super.key,
    required this.sessionId,
    required this.accId,
    required this.ledgerId,
    required this.onPayNow,
  });

  @override
  State<BillPendingToast> createState() => _BillPendingToastState();
}

class _BillPendingToastState extends State<BillPendingToast> {
  @override
  void initState() {
    super.initState();

    /// 🔹 Fetch data after one frame (safe after build)
    WidgetsBinding.instance.addPostFrameCallback((_) {
      final provider = Provider.of<TransactionsProvider>(context, listen: false);
      provider.fetchPaymentReceiptDetails(
        widget.sessionId,
        widget.accId,
        widget.ledgerId,
      );
    });
  }


  @override
  Widget build(BuildContext context) {
    final provider = Provider.of<TransactionsProvider>(context);
    final data = provider.receiptDetails;

    return Dialog(
      insetPadding: const EdgeInsets.symmetric(horizontal: 24),
      backgroundColor: Colors.white,
      shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(16)),
      child: Container(
        width: 360,
        padding: const EdgeInsets.all(20),
        decoration: BoxDecoration(
          borderRadius: BorderRadius.circular(16),
          color: Colors.white,
        ),
        child: provider.isLoading
            ? const SizedBox(
            height: 200,
            child: Center(child: CircularProgressIndicator()))
            : Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            // 🔴 Red Pending Icon
            Container(
              height: 60,
              width: 60,
              decoration: const BoxDecoration(
                color: Color(0xFFFFEBEB),
                shape: BoxShape.circle,
              ),
              child: const Icon(
                Icons.access_time_filled,
                color: Colors.redAccent,
                size: 40,
              ),
            ),
            const SizedBox(height: 16),

            // Title + Amount
            const Text(
              "Bill Pending",
              style: TextStyle(
                fontSize: 18,
                fontFamily: "Poppins",
                fontWeight: FontWeight.w600,
                color: Colors.black,
              ),
            ),
            const SizedBox(height: 8),
            Text(
              "₹${data?.dAmount ?? "--"}",
              style: const TextStyle(
                fontSize: 28,
                fontWeight: FontWeight.bold,
                fontFamily: "Poppins",
                color: Colors.black,
              ),
            ),

            const SizedBox(height: 12),
            const Divider(thickness: 1, color: Color(0xFFE6E6E6)),
            const SizedBox(height: 12),

            // Payment Details
            const Align(
              alignment: Alignment.centerLeft,
              child: Text(
                "Payment Details",
                style: TextStyle(
                  fontSize: 14,
                  fontWeight: FontWeight.w600,
                  fontFamily: "Poppins",
                  color: Colors.black,
                ),
              ),
            ),
            const SizedBox(height: 8),
            _buildRow("Order ID", "#${data?.id ?? "--"}", Colors.blue),
            _buildRow("Date", data?.prDate ?? "--", Colors.black),

            const SizedBox(height: 12),
            const Divider(thickness: 1, color: Color(0xFFE6E6E6)),
            const SizedBox(height: 12),

            // Products Section
            const Align(
              alignment: Alignment.centerLeft,
              child: Text(
                "Products",
                style: TextStyle(
                  fontSize: 14,
                  fontWeight: FontWeight.w600,
                  fontFamily: "Poppins",
                  color: Colors.black,
                ),
              ),
            ),
            const SizedBox(height: 8),
            _buildRow(
              data?.narration ?? "Product Item",
              "₹${data?.dAmount ?? "--"}",
              Colors.black,
            ),

            const SizedBox(height: 12),
            const Divider(thickness: 1, color: Color(0xFFE6E6E6)),
            const SizedBox(height: 8),

            // SubTotal + Gross + Tax + Total
            _buildRow(
                "Sub Total", "₹${data?.dAmount ?? "--"}", Colors.black),
            _buildRow("Gross Amount",
                "₹${data?.cAmount ?? "--"}", Colors.black),
            _buildRow("SGST", "0", Colors.black),
            _buildRow("CGST", "0", Colors.black),
            _buildRow("IGST", "0", Colors.black),

            const Divider(thickness: 1, color: Color(0xFFE6E6E6)),
            const SizedBox(height: 4),
            Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                const Text(
                  "Total",
                  style: TextStyle(
                    fontSize: 15,
                    fontFamily: "Poppins",
                    fontWeight: FontWeight.w600,
                    color: Colors.black,
                  ),
                ),
                Text(
                  "₹${data?.cAmount ?? "--"}",
                  style: const TextStyle(
                    fontSize: 15,
                    fontFamily: "Poppins",
                    fontWeight: FontWeight.w600,
                    color: Colors.black,
                  ),
                ),
              ],
            ),

            const SizedBox(height: 24),

            // Buttons Row
            Row(
              children: [
                Expanded(
                  child: OutlinedButton.icon(
                    onPressed: provider.isDownloading
                        ? null
                        : () async {
                      await provider.downloadPaymentReceipt(
                        context,
                        widget.sessionId,
                        widget.ledgerId,
                        widget.accId,
                      );
                    },
                    icon: provider.isDownloading
                        ? const SizedBox(
                      width: 16,
                      height: 16,
                      child: CircularProgressIndicator(
                        strokeWidth: 2,
                        color: Colors.black54,
                      ),
                    )
                        : const Icon(Icons.download_rounded,
                        color: Colors.black),
                    label: const Text(
                      "Download",
                      style: TextStyle(
                        fontSize: 14,
                        fontFamily: "Poppins",
                        fontWeight: FontWeight.w600,
                        color: Colors.black,
                      ),
                    ),
                    style: OutlinedButton.styleFrom(
                      side: const BorderSide(color: Colors.black26),
                      shape: RoundedRectangleBorder(
                        borderRadius: BorderRadius.circular(50),
                      ),
                      padding: const EdgeInsets.symmetric(vertical: 14),
                    ),
                  ),
                ),
                const SizedBox(width: 12),
                Expanded(
                  child: ElevatedButton(
                    onPressed: widget.onPayNow,
                    style: ElevatedButton.styleFrom(
                      backgroundColor: AppColors.amountText,
                      shape: RoundedRectangleBorder(
                        borderRadius: BorderRadius.circular(50),
                      ),
                      padding: const EdgeInsets.symmetric(vertical: 14),
                    ),
                    child: const Text(
                      "Pay Now",
                      style: TextStyle(
                        fontSize: 15,
                        fontFamily: "Poppins",
                        fontWeight: FontWeight.w600,
                        color: Colors.white,
                      ),
                    ),
                  ),
                ),
              ],
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildRow(String title, String value, Color valueColor) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 3),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Text(
            title,
            style: const TextStyle(
              fontSize: 14,
              fontFamily: "Poppins",
              color: Colors.black54,
            ),
          ),
          Text(
            value,
            style: TextStyle(
              fontSize: 14,
              fontFamily: "Poppins",
              fontWeight: FontWeight.w500,
              color: valueColor,
            ),
          ),
        ],
      ),
    );
  }
}
