import 'package:flutter/material.dart';
import 'package:flutter_svg/flutter_svg.dart';
import 'package:gen_rentals/Utility/AppColors.dart';

enum BillStatusType { paid, pending }

class BillStatusToast extends StatelessWidget {
  final BillStatusType type;
  final String amount;
  final String orderId;
  final String date;
  final String product;
  final String productPrice;
  final String total;
  final VoidCallback onDownload;
  final VoidCallback? onPayNow;

  const BillStatusToast({
    super.key,
    required this.type,
    required this.amount,
    required this.orderId,
    required this.date,
    required this.product,
    required this.productPrice,
    required this.total,
    required this.onDownload,
    this.onPayNow,
  });

  @override
  Widget build(BuildContext context) {
    final isPaid = type == BillStatusType.paid;
    return Dialog(
      insetPadding: const EdgeInsets.all(16),
      shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(16)),
      child: Padding(
        padding: const EdgeInsets.all(20),
        child: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            // ✅ or ❌ icon
            Container(
              decoration: BoxDecoration(
                color: isPaid
                    ? Colors.green.withOpacity(0.1)
                    : Colors.red.withOpacity(0.1),
                shape: BoxShape.circle,
              ),
              padding: const EdgeInsets.all(12),
              child: Icon(
                isPaid ? Icons.check_circle : Icons.access_time_filled,
                color: isPaid ? Colors.green : Colors.red,
                size: 40,
              ),
            ),
            const SizedBox(height: 12),

            Text(
              isPaid ? "Payment Receipt" : "Bill Pending",
              style: const TextStyle(
                fontSize: 18,
                fontWeight: FontWeight.w600,
                fontFamily: "Plus Jakarta Sans",
              ),
            ),
            const SizedBox(height: 6),

            Text(
              "₹$amount",
              style: const TextStyle(
                fontSize: 28,
                fontWeight: FontWeight.w700,
                color: Colors.black,
              ),
            ),
            const SizedBox(height: 18),

            // Details
            const Divider(thickness: 1),
            Align(
              alignment: Alignment.centerLeft,
              child: Text(
                "Payment Details",
                style: TextStyle(
                  fontSize: 15,
                  color: Colors.grey[700],
                  fontWeight: FontWeight.w500,
                ),
              ),
            ),
            const SizedBox(height: 10),

            _detailRow("Order ID", "#$orderId", true),
            _detailRow("Date", date, false),
            if (isPaid) _detailRow("Payment Mode", "UPI", false),

            const SizedBox(height: 10),
            const Divider(thickness: 1),

            Align(
              alignment: Alignment.centerLeft,
              child: Text(
                "Products",
                style: TextStyle(
                  fontSize: 15,
                  color: Colors.grey[700],
                  fontWeight: FontWeight.w500,
                ),
              ),
            ),
            const SizedBox(height: 10),

            _detailRow(product, "₹${productPrice}", false),
            const SizedBox(height: 10),
            const Divider(thickness: 1),

            _detailRow("Total", "₹$total", true),
            const SizedBox(height: 20),

            // Bottom buttons
            if (isPaid)
              SizedBox(
                width: double.infinity,
                child: ElevatedButton.icon(
                  onPressed: onDownload,
                  icon: const Icon(Icons.download, color: Colors.white),
                  label: const Text(
                    "Download Receipt",
                    style: TextStyle(
                      fontSize: 16,
                      fontFamily: "Plus Jakarta Sans",
                      fontWeight: FontWeight.w600,
                      color: Colors.white,
                    ),
                  ),
                  style: ElevatedButton.styleFrom(
                    backgroundColor: AppColors.amountText,
                    shape: RoundedRectangleBorder(
                      borderRadius: BorderRadius.circular(50),
                    ),
                    padding: const EdgeInsets.symmetric(vertical: 14),
                  ),
                ),
              )
            else
              Row(
                children: [
                  Expanded(
                    child: OutlinedButton.icon(
                      onPressed: onDownload,
                      icon: const Icon(Icons.download, color: Colors.black),
                      label: const Text(
                        "Download",
                        style: TextStyle(
                          color: Colors.black,
                          fontWeight: FontWeight.w600,
                        ),
                      ),
                      style: OutlinedButton.styleFrom(
                        shape: RoundedRectangleBorder(
                          borderRadius: BorderRadius.circular(50),
                        ),
                        side: const BorderSide(color: Colors.black26),
                        padding: const EdgeInsets.symmetric(vertical: 14),
                      ),
                    ),
                  ),
                  const SizedBox(width: 10),
                  Expanded(
                    child: ElevatedButton(
                      onPressed: onPayNow,
                      style: ElevatedButton.styleFrom(
                        backgroundColor: AppColors.amountText,
                        shape: RoundedRectangleBorder(
                          borderRadius: BorderRadius.circular(50),
                        ),
                        padding: const EdgeInsets.symmetric(vertical: 14),
                      ),
                      child: const Text(
                        "Pay Now",
                        style: TextStyle(
                          fontSize: 16,
                          fontFamily: "Plus Jakarta Sans",
                          fontWeight: FontWeight.w600,
                          color: Colors.white,
                        ),
                      ),
                    ),
                  ),
                ],
              ),
          ],
        ),
      ),
    );
  }

  Widget _detailRow(String title, String value, bool highlight) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 4),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Text(title,
              style: const TextStyle(
                fontSize: 14,
                color: Colors.black54,
              )),
          Text(
            value,
            style: TextStyle(
              fontSize: 14,
              fontWeight: highlight ? FontWeight.w600 : FontWeight.w500,
              color: highlight ? Colors.black : Colors.black87,
            ),
          ),
        ],
      ),
    );
  }
}
