import 'package:dotted_line/dotted_line.dart';
import 'package:flutter/material.dart';
import 'package:flutter_svg/svg.dart';
import 'package:generp/Utils/custom_snackbar.dart';
import 'package:geocoding/geocoding.dart';
import 'package:provider/provider.dart';
import 'package:url_launcher/url_launcher.dart';

import '../../Notifiers/hrmProvider/attendanceDetailsProvider.dart';
import '../../Notifiers/HomeScreenNotifier.dart';
import '../../Utils/app_colors.dart';
import '../finance/FileViewer.dart';

/// screen for attendance details
class AttendanceRequestDetailScreen extends StatefulWidget {
  final String mode;
  final attendanceListId;
  const AttendanceRequestDetailScreen({
    super.key,
    required this.attendanceListId,
    required this.mode,
  });

  @override
  State<AttendanceRequestDetailScreen> createState() =>
      _AttendanceRequestDetailScreenState();
}

class _AttendanceRequestDetailScreenState
    extends State<AttendanceRequestDetailScreen> {
  final bool _actionSubmitted = false;
  late AttendanceDetailsProvider provider;

  @override
  Widget build(BuildContext context) {
    return SafeArea(
      top: false,
      child: ChangeNotifierProvider(
        create:
            (_) =>
                AttendanceDetailsProvider()..fetchAttendanceRequestDetail(
                  context,
                  widget.attendanceListId,
                ),
        child: Consumer<AttendanceDetailsProvider>(
          builder: (context, provider, child) {
            // Get screen dimensions for responsive scaling
            final screenWidth = MediaQuery.of(context).size.width;
            final screenHeight = MediaQuery.of(context).size.height;

            // Scale factors based on screen size
            final scaleFactor = screenWidth / 360; // Base width for scaling
            final textScaleFactor = MediaQuery.of(
              context,
            ).textScaleFactor.clamp(1.0, 1.2);

            return Scaffold(
              appBar: AppBar(
                automaticallyImplyLeading: false,
                backgroundColor: Color(0xFFFFFFFF),
                title: Row(
                  mainAxisAlignment: MainAxisAlignment.start,
                  crossAxisAlignment: CrossAxisAlignment.center,
                  children: [
                    InkResponse(
                      onTap: () => Navigator.pop(context, true),
                      child: SvgPicture.asset(
                        "assets/svg/appbar_back_button.svg",
                        height: 25 * scaleFactor,
                      ),
                    ),
                    SizedBox(width: 10 * scaleFactor),
                    InkResponse(
                      onTap: () => Navigator.pop(context, true),
                      child: Text(
                        "Attendance Details",
                        style: TextStyle(
                          fontSize: 18,
                          height: 1.1,
                          fontFamily: "Plus Jakarta Sans",
                          fontWeight: FontWeight.w600,
                          color: AppColors.semi_black,
                        ),
                      ),
                    ),
                  ],
                ),
              ),
              backgroundColor: Color(0xFFF6F6F8),
              body: Builder(
                builder: (context) {
                  if (provider.isLoading) {
                    return const Center(
                      child: CircularProgressIndicator(color: Colors.blue),
                    );
                  }
                  // if (provider.errorMessage != null) {
                  //   return Center(child: Text(provider.errorMessage!));
                  // }
                  if (provider.response?.requestDetails == null) {
                    return const Center(child: Text("No details found"));
                  }

                  final details = provider.response!.requestDetails!;
                  final checkInTime =
                      details.status == "Requested"
                          ? "--"
                          : details.checkInTime;

                  /// scr
                  return SingleChildScrollView(
                    padding: EdgeInsets.all(16.0 * scaleFactor),
                    child: Column(
                      children: [
                        Card(
                          shape: RoundedRectangleBorder(
                            borderRadius: BorderRadius.circular(
                              16 * scaleFactor,
                            ),
                          ),
                          elevation: 0,
                          child: Padding(
                            padding: EdgeInsets.all(16.0 * scaleFactor),
                            child: Column(
                              crossAxisAlignment: CrossAxisAlignment.start,
                              children: [
                                Container(
                                  margin: EdgeInsets.only(
                                    bottom: 0.5 * scaleFactor,
                                  ),
                                  padding: EdgeInsets.symmetric(
                                    horizontal: 2.5 * scaleFactor,
                                    vertical: 12 * scaleFactor,
                                  ),
                                  decoration: BoxDecoration(
                                    color: Colors.white,
                                    borderRadius: BorderRadius.circular(
                                      12 * scaleFactor,
                                    ),
                                  ),
                                  child: Row(
                                    children: [
                                      /// Left Avatar
                                      Container(
                                        height: 44 * scaleFactor,
                                        width: 44 * scaleFactor,
                                        decoration: BoxDecoration(
                                          shape: BoxShape.circle,
                                          color: const Color(
                                            0xFFEDF8FF,
                                          ), // icon bg
                                        ),
                                        child: Center(
                                          child: SvgPicture.asset(
                                            height: 24 * scaleFactor,
                                            width: 24 * scaleFactor,
                                            "assets/svg/hrm/attendanceList.svg",
                                            fit: BoxFit.contain,
                                          ),
                                        ),
                                      ),
                                      SizedBox(width: 12 * scaleFactor),

                                      /// Middle text
                                      Expanded(
                                        child: Column(
                                          crossAxisAlignment:
                                              CrossAxisAlignment.start,
                                          children: [
                                            Text(
                                              details.type ?? "-",
                                              style: TextStyle(
                                                decoration:
                                                    TextDecoration.underline,
                                                decorationStyle:
                                                    TextDecorationStyle.dotted,
                                                decorationColor:
                                                    AppColors.grey_thick,
                                                height: 1.2,
                                                fontFamily: "JakartaRegular",
                                                fontSize: 14,
                                                color: AppColors.semi_black,
                                              ),
                                            ),
                                            SizedBox(height: 2 * scaleFactor),
                                            Text(
                                              details.date ?? "-",
                                              style: TextStyle(
                                                fontFamily: "JakartaRegular",
                                                fontSize: 14,
                                                color: AppColors.app_blue,
                                              ),
                                            ),
                                          ],
                                        ),
                                      ),

                                      /// Right side (Live/Manual)
                                      Container(
                                        height: 30 * scaleFactor,
                                        padding: EdgeInsets.symmetric(
                                          horizontal: 12 * scaleFactor,
                                        ),
                                        decoration: BoxDecoration(
                                          borderRadius: BorderRadius.circular(
                                            6 * scaleFactor,
                                          ),
                                          color: getDecorationColor(
                                            details.status,
                                          ),
                                        ),
                                        child: Center(
                                          child: Text(
                                            details.status ?? "-",
                                            style: TextStyle(
                                              fontSize: 13,
                                              fontWeight: FontWeight.w600,
                                              color: getTextColor(
                                                details.status.toString(),
                                              ),
                                            ),
                                          ),
                                        ),
                                      ),
                                    ],
                                  ),
                                ),

                                // Employee Details
                                _buildSectionHeader(
                                  "Employee Details",
                                  scaleFactor,
                                ),
                                _buildDetailTile(
                                  "Employee Name",
                                  details.employeeName,
                                  scaleFactor,
                                ),
                                _buildDetailTile(
                                  "Created Employee",
                                  details.createdEmpName,
                                  scaleFactor,
                                ),

                                // Check In/Out
                                _buildSectionHeader(
                                  "Check In/Out Details",
                                  scaleFactor,
                                ),
                                if (details.type == "Check In" || details.type == "Check In/Out")
                                _buildDate_TimeTile(
                                  "Check In Date & Time",
                                  details.date,
                                  details.checkInTime,
                                  scaleFactor,
                                ),
                                if (details.type == "Check Out" || details.type == "Check In/Out")
                                _buildDate_TimeTile(
                                  "Check Out Date & Time",
                                  details.date,
                                  details.checkOutTime,
                                  scaleFactor,
                                ),

                                if (details.type == "Check In" || details.type == "Check In/Out")
                                _buildDetailTile(
                                  "Original Check In",
                                  checkInTime,
                                  scaleFactor,
                                ),

                                if (details.type == "Check Out" || details.type == "Check In/Out")
                                _buildDetailTile(
                                  "Original Check Out",
                                  "--",
                                  scaleFactor,
                                ),

                                if (details.type == "Check In" || details.type == "Check In/Out")
                                _buildDetailTile(
                                  "Original Check In Location",
                                  details.checkInLocation,
                                  scaleFactor,
                                ),

                                if (details.type == "Check Out" || details.type == "Check In/Out")
                                _buildDetailTile(
                                  "Original Check Out Location",
                                  details.checkOutLocation,
                                  scaleFactor,
                                ),
                                buildLocationTile(
                                  "Location",
                                  details.location,
                                  scaleFactor,
                                ),

                                // Proofs
                                if ((details.checkInProofDirFilePath != null &&
                                        details
                                            .checkInProofDirFilePath!
                                            .isNotEmpty) ||
                                    (details.checkOutProofDirFilePath != null &&
                                        details
                                            .checkOutProofDirFilePath!
                                            .isNotEmpty)) ...[
                                  _buildSectionHeader("Proofs", scaleFactor),
                                  if (details.checkInProofDirFilePath != null &&
                                      details
                                          .checkInProofDirFilePath!
                                          .isNotEmpty)
                                    _buildProofLink(
                                      context,
                                      "Check In Proof",
                                      details.checkInProofDirFilePath,
                                      scaleFactor,
                                    ),
                                  if (details.checkOutProofDirFilePath !=
                                          null &&
                                      details
                                          .checkOutProofDirFilePath!
                                          .isNotEmpty)
                                    _buildProofLink(
                                      context,
                                      "Check Out Proof",
                                      details.checkOutProofDirFilePath,
                                      scaleFactor,
                                    ),
                                ],

                                // Remarks & Approvals
                                _buildSectionHeader(
                                  "Remarks & Approvals",
                                  scaleFactor,
                                ),
                                _buildDetailTile(
                                  "Level 1 Approved By",
                                  details.level1EmpName,
                                  scaleFactor,
                                ),
                                _buildDetailTile(
                                  "Level 2 Approved By",
                                  details.level2EmpName,
                                  scaleFactor,
                                ),
                                _buildDetailTile(
                                  "Level 1 Remark",
                                  details.level1Remarks,
                                  scaleFactor,
                                ),
                                _buildDetailTile(
                                  "Level 2 Remark",
                                  details.level2Remarks,
                                  scaleFactor,
                                ),

                                ///remain data
                                _buildSectionHeader(
                                  "Other Details",
                                  scaleFactor,
                                ),
                                if (details.type == "Check In" || details.type == "Check In/Out")
                                _buildDetailTile(
                                  "Check In Type",
                                  details.checkInType,
                                  scaleFactor,
                                ),
                                if (details.type == "Check Out" || details.type == "Check In/Out")
                                _buildDetailTile(
                                  "Check Out Type",
                                  details.chechOutType,
                                  scaleFactor,
                                ),
                                if (details.type == "Check Out" || details.type == "Check In/Out")
                                _buildDetailTile(
                                  "Check Out Time",
                                  details.checkOutTime,
                                  scaleFactor,
                                ),
                                // Attendance Info
                                _buildDetailTile("ID", details.id, scaleFactor),
                                _buildDetailTile(
                                  "Attendance Type",
                                  details.attendanceType,
                                  scaleFactor,
                                ),
                                _buildDetailTile(
                                  "Note",
                                  details.note,
                                  scaleFactor,
                                ),
                                _buildDetailTile(
                                  "Created Datetime",
                                  details.requestedDatetime,
                                  scaleFactor,
                                ),
                              ],
                            ),
                          ),
                        ),
                        SizedBox(height: 30 * scaleFactor),
                      ],
                    ),
                  );
                },
              ),
              bottomNavigationBar:
                  (widget.mode == "apr_lvl1" &&
                          !_actionSubmitted &&
                          provider.response?.requestDetails?.status !=
                              "Level 1 Approved" &&
                          provider.response?.requestDetails?.status !=
                              "Level 1 Rejected")
                      ? Container(
                        decoration: const BoxDecoration(
                          gradient: LinearGradient(
                            begin: Alignment.topCenter,
                            end: Alignment.bottomCenter,
                            colors: [Color(0xffFFFFFF), Color(0x00FFFFFF)],
                          ),
                        ),
                        padding: const EdgeInsets.symmetric(
                          horizontal: 10,
                          vertical: 6,
                        ),
                        height: 61,
                        child: Column(
                          children: [
                            Row(
                              children: [
                                /// Reject Button
                                Expanded(
                                  child: InkWell(
                                    onTap: () {
                                      showRemarkSheet(
                                        context: context,
                                        actionType: "Reject",
                                        onSubmit: (remark) {
                                          provider
                                              .rejectApproveAttendanceRequest(
                                                context,
                                                mode: widget.mode,
                                                type: "Rejected",
                                                remarks: remark,
                                                id:
                                                    provider
                                                        .response!
                                                        .requestDetails!
                                                        .id!,
                                              );
                                        },
                                      ).then((_) {
                                        provider.fetchAttendanceRequestDetail(
                                          context,
                                          widget.attendanceListId,
                                        ); // or setState(() {}) if needed
                                      });
                                    },
                                    child: Container(
                                      alignment: Alignment.center,
                                      decoration: BoxDecoration(
                                        borderRadius: BorderRadius.circular(8),
                                      ),
                                      child: Row(
                                        mainAxisAlignment:
                                            MainAxisAlignment.center,
                                        children: [
                                          SvgPicture.asset(
                                            "assets/svg/finance/level_reject_ic.svg",
                                          ),
                                          const SizedBox(width: 6),
                                          const Text("Reject"),
                                        ],
                                      ),
                                    ),
                                  ),
                                ),

                                /// Vertical Divider
                                Container(
                                  width: 1,
                                  height: 45,
                                  color: Colors.grey.shade300,
                                ),

                                /// Approve Button
                                Expanded(
                                  child: InkWell(
                                    onTap: () {
                                      showRemarkSheet(
                                        context: context,
                                        actionType: "Approve",
                                        onSubmit: (remark) async {
                                          await provider
                                              .rejectApproveAttendanceRequest(
                                                context,
                                                mode: widget.mode,
                                                type: "Approved",
                                                remarks: remark,
                                                id:
                                                    provider
                                                        .response!
                                                        .requestDetails!
                                                        .id!,
                                              );
                                        },
                                      ).then((_) {
                                        provider.fetchAttendanceRequestDetail(
                                          context,
                                          widget.attendanceListId,
                                        ); // or setState(() {}) if needed
                                      });
                                    },

                                    child: Container(
                                      alignment: Alignment.center,
                                      decoration: BoxDecoration(
                                        borderRadius: BorderRadius.circular(8),
                                      ),
                                      child: Row(
                                        mainAxisAlignment:
                                            MainAxisAlignment.center,
                                        children: [
                                          SvgPicture.asset(
                                            "assets/svg/finance/level_approve_ic.svg",
                                          ),
                                          const SizedBox(width: 6),
                                          const Text("Approve"),
                                        ],
                                      ),
                                    ),
                                  ),
                                ),
                              ],
                            ),
                            SizedBox(height: 0),
                          ],
                        ),
                      )
                      : const SizedBox.shrink(),
              floatingActionButtonLocation:
                  FloatingActionButtonLocation.centerDocked,
            );
          },
        ),
      ),
    );
  }

  Future<void> showRemarkSheet({
    required BuildContext context,
    required String actionType, // "Approved" or "Rejected"
    required Function(String remark) onSubmit,
  }) {
    final remarkController = TextEditingController();
    String? remarkError;

    return showModalBottomSheet(
      useSafeArea: true,
      isDismissible: true,
      isScrollControlled: true,
      showDragHandle: true,
      backgroundColor: Colors.white,
      enableDrag: true,
      context: context,
      builder: (context) {
        return StatefulBuilder(
          builder: (context, setState) {
            void updateState(VoidCallback fn) {
              setState(fn);
            }

            bool validateFields() {
              String? newRemarkError =
                  remarkController.text.trim().isEmpty
                      ? "Remark required"
                      : null;

              if (remarkError != newRemarkError) {
                updateState(() {
                  remarkError = newRemarkError;
                });
              }
              return newRemarkError == null;
            }

            Widget errorText(String? msg) =>
                msg == null
                    ? const SizedBox()
                    : Padding(
                      padding: const EdgeInsets.only(top: 4, left: 4),
                      child: Text(
                        msg,
                        style: const TextStyle(
                          color: Colors.red,
                          fontSize: 12,
                          fontFamily: "JakartaMedium",
                        ),
                      ),
                    );

            return SafeArea(
              child: Container(
                margin: const EdgeInsets.symmetric(
                  horizontal: 15,
                  vertical: 10,
                ),
                padding: EdgeInsets.only(
                  bottom: MediaQuery.of(context).viewInsets.bottom,
                ),
                child: SingleChildScrollView(
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    mainAxisSize: MainAxisSize.min,
                    children: [
                      Text(
                        "$actionType Attendance Request",
                        style: const TextStyle(
                          fontSize: 16,
                          color: Colors.black87,
                          fontFamily: "JakartaMedium",
                        ),
                      ),
                      const SizedBox(height: 16),

                      Text(
                        "Remark",
                        style: const TextStyle(
                          fontSize: 14,
                          color: Colors.black87,
                          fontFamily: "JakartaMedium",
                        ),
                      ),
                      const SizedBox(height: 6),

                      Container(
                        margin: const EdgeInsets.only(bottom: 6),
                        decoration: BoxDecoration(
                          color: AppColors.text_field_color,
                          borderRadius: BorderRadius.circular(14),
                        ),
                        child: TextField(
                          controller: remarkController,
                          maxLines: 3,
                          onChanged: (val) {
                            if (remarkError != null && val.isNotEmpty) {
                              updateState(() => remarkError = null);
                            }
                          },
                          decoration: InputDecoration(
                            hintText: "Enter your remark here...",
                            hintStyle: TextStyle(
                              color:
                                  Colors.grey.shade500, // Customize this color
                              fontSize: 14, // Optional: tweak font size
                            ),
                            border: InputBorder.none,
                            contentPadding: const EdgeInsets.symmetric(
                              horizontal: 12,
                              vertical: 12,
                            ),
                          ),
                        ),
                      ),
                      errorText(remarkError),

                      const SizedBox(height: 20),

                      Row(
                        children: [
                          Expanded(
                            child: InkResponse(
                              onTap: () => Navigator.pop(context),
                              child: Container(
                                height: 45,
                                decoration: BoxDecoration(
                                  color: Color(0x12AAAAAA),
                                  borderRadius: BorderRadius.circular(12),
                                ),
                                child: const Center(
                                  child: Text(
                                    "Cancel",
                                    style: TextStyle(
                                      color: Colors.red,
                                      fontFamily: "JakartaMedium",
                                    ),
                                  ),
                                ),
                              ),
                            ),
                          ),
                          const SizedBox(width: 12),
                          Expanded(
                            child: InkResponse(
                              onTap: () async {
                                if (validateFields()) {
                                  final remark = remarkController.text.trim();
                                  // Call provider
                                  await onSubmit(remark);
                                  //  SnackBar here
                                  Navigator.pop(context);
                                  CustomSnackBar.showSuccess(
                                      context: context,
                                      message: "Request submitted successfully"
                                  );

                                }
                              },
                              child: Container(
                                height: 45,
                                decoration: BoxDecoration(
                                  color: Colors.blue,
                                  borderRadius: BorderRadius.circular(12),
                                ),
                                child: const Center(
                                  child: Text(
                                    "Submit",
                                    style: TextStyle(
                                      color: Colors.white,
                                      fontFamily: "JakartaMedium",
                                    ),
                                  ),
                                ),
                              ),
                            ),
                          ),
                        ],
                      ),
                    ],
                  ),
                ),
              ),
            );
          },
        );
      },
    );
  }

  ///  Reusable Row Widget for details
  Widget _buildDetailTile(String label, String? value, double scaleFactor) {
    return Padding(
      padding: EdgeInsets.symmetric(vertical: 3 * scaleFactor),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start, // Align top if wraps
        children: [
          // Label
          Expanded(
            flex: 5, // keep same ratio as other tiles
            child: Text(
              label,
              style: TextStyle(
                fontFamily: "JakartaRegular",
                fontSize: 14,
                color: AppColors.semi_black,
              ),
            ),
          ),
          SizedBox(width: 4),

          // Value
          Expanded(
            flex: 5, // take remaining width
            child: Text(
              value ?? "-",
              style: const TextStyle(fontSize: 14, color: Color(0xFF818181)),
              softWrap: true,
              overflow: TextOverflow.visible, // wrap instead of clipping
            ),
          ),
        ],
      ),
    );
  }

  /// for location
  /// Location Tile
  Widget buildLocationTile(String label, String? value, double scaleFactor) {
    return FutureBuilder<String>(
      future: getReadableLocation(value),
      builder: (context, snapshot) {
        final locationText =
            snapshot.connectionState == ConnectionState.done
                ? (snapshot.data ?? value ?? "-")
                : value ?? "-";

        return Padding(
          padding: EdgeInsets.symmetric(vertical: 6 * scaleFactor),
          child: Row(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              // Label
              Expanded(
                flex: 5,
                child: Text(
                  label,
                  style: TextStyle(
                    fontFamily: "JakartaRegular",
                    fontSize: 14,
                    color: AppColors.semi_black,
                  ),
                ),
              ),

              // Value (Clickable Location)
              Expanded(
                flex: 5,
                child: GestureDetector(
                  onTap: () async {
                    final uri = Uri.parse(
                      "https://www.google.com/maps/search/?api=1&query=$value",
                    );
                    if (await canLaunchUrl(uri)) {
                      await launchUrl(
                        uri,
                        mode: LaunchMode.externalApplication,
                      );
                    }
                  },
                  child: Text(
                    locationText,
                    style: const TextStyle(
                      fontSize: 14,
                      color: Colors.blue,
                      decoration: TextDecoration.underline,
                      fontWeight: FontWeight.w400,
                    ),
                    softWrap: true,
                    overflow: TextOverflow.visible,
                  ),
                ),
              ),
            ],
          ),
        );
      },
    );
  }

  /// Convert coordinates -> human readable location
  /// Convert coordinates -> full human readable location
  Future<String> getReadableLocation(String? value) async {
    if (value == null || value.isEmpty) return "-";
    try {
      // Expecting "lat,lng"
      final parts = value.split(',');
      if (parts.length != 2) return value;

      final lat = double.tryParse(parts[0].trim());
      final lng = double.tryParse(parts[1].trim());
      if (lat == null || lng == null) return value;

      final placemarks = await placemarkFromCoordinates(lat, lng);
      final place = placemarks.first;

      // Include more details
      final address = [
        place.name,
        place.street, // e.g. "A-46, Lata Enclave"
        place.subLocality, // e.g. "Madhura Nagar"
        place.locality, // e.g. "Hyderabad"
        place.administrativeArea, // e.g. "Telangana"
        place.postalCode, // e.g. "500038"
        place.country, // e.g. "India"
      ].where((e) => e != null && e.isNotEmpty).join(", ");

      return address;
    } catch (e) {
      return value; // fallback to raw coordinates
    }
  }

  /// for date and time
  Widget _buildDate_TimeTile(
    String label,
    String? date,
    String? time,
    double scaleFactor,
  ) {
    return Padding(
      padding: EdgeInsets.symmetric(vertical: 6 * scaleFactor),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start, // align top when wrapped
        children: [
          // Label
          Expanded(
            flex: 5, // adjust ratio
            child: Text(
              label,
              style: TextStyle(
                fontFamily: "JakartaRegular",
                fontSize: 14,
                color: AppColors.semi_black,
              ),
            ),
          ),
          SizedBox(width: 4),

          // Value (date + time)
          Expanded(
            flex: 5, // adjust ratio so both fill row
            child: Text(
              '${date ?? "-"}, ${time ?? "-"}',
              style: const TextStyle(
                fontSize: 14,
                color: Color(0xff818181),
                fontWeight: FontWeight.w400,
              ),
              softWrap: true, // allow wrapping
              overflow: TextOverflow.visible,
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildSectionHeader(String title, double scaleFactor) {
    return Padding(
      padding: EdgeInsets.symmetric(vertical: 9 * scaleFactor),
      child: Row(
        children: [
          Text(
            title,
            style: TextStyle(fontSize: 14, fontFamily: "JakartaSemiBold"),
          ),
          SizedBox(width: 10 * scaleFactor),
          Expanded(
            child: DottedLine(
              dashGapLength: 4,
              dashGapColor: Colors.white,
              dashColor: AppColors.grey_semi,
              dashLength: 2,
              lineThickness: 0.5,
            ),
          ),
        ],
      ),
    );
  }

  ///  Proof section (image/file path)
  Widget _buildProofLink(
    BuildContext context,
    String label,
    String? filePath,
    double scaleFactor,
  ) {
    return Padding(
      padding: EdgeInsets.symmetric(vertical: 6 * scaleFactor),
      child: Row(
        children: [
          Expanded(
            flex: 5,
            child: Text(
              label,
              style: TextStyle(
                fontFamily: "JakartaRegular",
                fontSize: 14,
                color: AppColors.semi_black,
              ),
            ),
          ),
          Expanded(
            flex: 0,
            child:
                filePath != null
                    ? InkWell(
                      onTap: () {
                        print("++++++++++++++++ImageUrel: $filePath");
                        Navigator.push(
                          context,
                          MaterialPageRoute(
                            builder:
                                (context) => Fileviewer(
                                  fileName: filePath ?? "",
                                  fileUrl: filePath ?? "",
                                  downloadEnable: false,
                                ),
                          ),
                        );
                      },
                      child: const Text(
                        "View",
                        style: TextStyle(
                          fontSize: 14,
                          color: Colors.blue,
                          fontStyle: FontStyle.normal,
                          fontFamily: "Plus Jakarta Sans",
                          fontWeight: FontWeight.w400,
                          decoration: TextDecoration.underline,
                        ),
                      ),
                    )
                    : const Text("-"),
          ),
        ],
      ),
    );
  }

  Color getTextColor(value) {
    var color = AppColors.approved_text_color;
    switch (value) {
      case 'Requested':
        return AppColors.requested_text_color;
      case 'Level 1 Approved':
        return AppColors.approved_text_color;
      case 'Level 1 Rejected':
        return AppColors.rejected_text_color;
      case 'Level 2 Approved':
        return AppColors.approved_text_color;
      case 'Level 2 Rejected':
        return AppColors.rejected_text_color;
      case 'Processed':
        return AppColors.processed_text_color;
      case 'Payment Rejected':
        return AppColors.rejected_text_color;
    }
    return color;
  }

  Color getDecorationColor(value) {
    var color = AppColors.approved_bg_color;
    switch (value) {
      case 'Requested':
        return AppColors.requested_bg_color;
      case 'Level 1 Approved':
        return AppColors.approved_bg_color;
      case 'Level 1 Rejected':
        return AppColors.rejected_bg_color;
      case 'Level 2 Approved':
        return AppColors.approved_bg_color;
      case 'Level 2 Rejected':
        return AppColors.rejected_bg_color;
      case 'Processed':
        return AppColors.processed_bg_color;
      case 'Payment Rejected':
        return AppColors.rejected_bg_color;
    }
    return color;
  }
}
