import 'package:dotted_line/dotted_line.dart';
import 'package:flutter/material.dart';
import 'package:flutter_svg/svg.dart';
import 'package:generp/screens/hrm/RewardSearchScreen.dart';
import 'package:provider/provider.dart';

import '../../Notifiers/hrmProvider/rewardListProvider.dart';
import '../../Utils/app_colors.dart';

class RewardListScreen extends StatefulWidget {
  const RewardListScreen({super.key});

  @override
  State<RewardListScreen> createState() => _RewardListScreenState();
}

class _RewardListScreenState extends State<RewardListScreen> {
  // Responsive text size function
  double getResponsiveTextSize(BuildContext context, double baseSize) {
    final double width = MediaQuery.of(context).size.width;

    if (width < 360) { // Small phones
      return baseSize * 0.85;
    } else if (width < 400) { // Medium phones
      return baseSize;
    } else { // Large phones
      return baseSize * 1.1;
    }
  }

  // Responsive padding function
  double getResponsivePadding(BuildContext context) {
    final double width = MediaQuery.of(context).size.width;
    return width * 0.04; // 4% of screen width
  }

  // Responsive height function
  double getResponsiveHeight(BuildContext context, double baseHeight) {
    final double height = MediaQuery.of(context).size.height;
    if (height < 700) { // Small height devices
      return baseHeight * 0.85;
    } else if (height < 800) { // Medium height devices
      return baseHeight;
    } else { // Large height devices
      return baseHeight * 1.15;
    }
  }

  @override
  Widget build(BuildContext context) {
    final bool isSmallScreen = MediaQuery.of(context).size.width < 360;
    final bool isLargeScreen = MediaQuery.of(context).size.width > 400;

    return SafeArea(
      top: false,
      child: ChangeNotifierProvider(
        create: (_) => RewardListProvider()..fetchRewardList(context),
        child: Consumer<RewardListProvider>(
          builder: (context, provider, child) {
            return Scaffold(
              appBar: AppBar(
                automaticallyImplyLeading: false,
                backgroundColor: Colors.white,
                title: Row(
                  children: [
                    InkResponse(
                      onTap: () => Navigator.pop(context, true),
                      child: SvgPicture.asset(
                        "assets/svg/appbar_back_button.svg",
                        height: isSmallScreen ? 22 : 25,
                      ),
                    ),
                    SizedBox(width: isSmallScreen ? 8 : 10),
                    Text(
                      "Reward List",
                      style: TextStyle(
                        fontSize: getResponsiveTextSize(context, 18),
                        fontFamily: "Plus Jakarta Sans",
                        fontWeight: FontWeight.w600,
                        color: AppColors.semi_black,
                      ),
                    ),
                  ],
                ),
                // actions: [
                //   InkResponse(
                //     onTap: () {
                //       Navigator.push(
                //         context,
                //         MaterialPageRoute(
                //           builder: (context) => RewardSearchScreen(),
                //           settings: const RouteSettings(
                //             name: 'AddLiveAttendanceScreen',
                //           ),
                //         ),
                //       ).then((_) {
                //       });
                //     },
                //     child: SvgPicture.asset(
                //       "assets/svg/search_ic.svg",
                //       height: isSmallScreen ? 22 : 25,
                //     ),
                //   ),
                //   SizedBox(width: isSmallScreen ? 15 : 20),
                // ],
              ),

              backgroundColor: Color(0xFFF6F6F8),
              body: Builder(
                builder: (context) {
                  if (provider.isLoading) {
                    return const Center(
                      child: CircularProgressIndicator(color: Colors.blue),
                    );
                  }
                  if (provider.errorMessage != null) {
                    return Center(
                      child: Text(
                        provider.errorMessage!,
                        style: TextStyle(
                          fontSize: getResponsiveTextSize(context, 14),
                        ),
                      ),
                    );
                  }
                  if (provider.response == null) {
                    return Center(
                      child: Text(
                        "No details found",
                        style: TextStyle(
                          fontSize: getResponsiveTextSize(context, 14),
                        ),
                      ),
                    );
                  }
                  final rewardDetail = provider.response!;
                  final rewardResponse = provider.response!;
                  final rewards =
                      rewardResponse.rewardsList; //  main list object
                  final achieved = rewardResponse.achievedAmount ?? "0";
                  final disbursed = rewardResponse.disbursedAmount ?? "0";
                  final balance = rewardResponse.balanceAmount ?? "0";
                  return SingleChildScrollView(
                    padding: EdgeInsets.all(getResponsivePadding(context)),
                    child: Column(
                      children: [
                        /// --- Top Summary Cards ---
                        Stack(
                          children: [
                            Container(
                              height: getResponsiveHeight(context, 110),
                              width: double.infinity,
                              padding: EdgeInsets.all(isSmallScreen ? 14 : 18),
                              decoration: BoxDecoration(
                                color: const Color(0xffd9ffd6),
                                borderRadius: BorderRadius.circular(18),
                              ),
                              child: Column(
                                crossAxisAlignment: CrossAxisAlignment.start,
                                mainAxisAlignment: MainAxisAlignment.center,
                                children: [
                                  Text(
                                    "₹$achieved", // Achieved Amount from response
                                    style: TextStyle(
                                      fontSize: getResponsiveTextSize(context, 20),
                                      color: const Color(0xff0D9C00),
                                      fontStyle: FontStyle.normal,
                                      fontWeight: FontWeight.w500,
                                    ),
                                  ),
                                  SizedBox(height: isSmallScreen ? 6 : 10),
                                  Text(
                                    "Achievement Amount",
                                    style: TextStyle(
                                      fontSize: getResponsiveTextSize(context, 14),
                                      color: const Color(0xff2D2D2D),
                                      fontStyle: FontStyle.normal,
                                      fontWeight: FontWeight.w400,
                                    ),
                                  ),
                                ],
                              ),
                            ),

                            // Positioned SVG Icon
                            Positioned(
                              bottom: isSmallScreen ? 6 : 8,
                              right: isSmallScreen ? 10 : 12,
                              child: Container(
                                height: isSmallScreen ? 36 : 42,
                                width: isSmallScreen ? 36 : 42,
                                decoration: BoxDecoration(
                                  shape: BoxShape.circle,
                                  color: const Color(0xA0FFFFFF), // icon bg
                                ),
                                child: Center(
                                  child: SvgPicture.asset(
                                    height: isSmallScreen ? 20 : 25,
                                    width: isSmallScreen ? 20 : 25,
                                    "assets/svg/hrm/achievement_ic.svg",
                                    fit: BoxFit.contain,
                                  ),
                                ),
                              ),
                            ),
                          ],
                        ),
                        SizedBox(height: isSmallScreen ? 8 : 12),

                        Row(
                          children: [
                            Expanded(
                              child: Container(
                                height: getResponsiveHeight(context, 110),
                                padding: EdgeInsets.all(isSmallScreen ? 12 : 16),
                                decoration: BoxDecoration(
                                  color: const Color(0xffe8ddff),
                                  borderRadius: BorderRadius.circular(16),
                                ),
                                child: Stack(
                                  children: [
                                    Column(
                                      crossAxisAlignment:
                                      CrossAxisAlignment.start,
                                      children: [
                                        Text(
                                          "₹$disbursed", // Disbursed Amount
                                          style: TextStyle(
                                            fontSize: getResponsiveTextSize(context, 20),
                                            color: const Color(0xff493272),
                                            fontWeight: FontWeight.w500,
                                          ),
                                        ),
                                        SizedBox(height: isSmallScreen ? 6 : 8),
                                        Text(
                                          "Disbursed \nAmount",
                                          style: TextStyle(
                                            fontSize: getResponsiveTextSize(context, 14),
                                            color: const Color(0xff2D2D2D),
                                            fontWeight: FontWeight.w400,
                                          ),
                                        ),
                                      ],
                                    ),
                                    Positioned(
                                      bottom: 2,
                                      right: 2,
                                      child: Container(
                                        height: isSmallScreen ? 36 : 42,
                                        width: isSmallScreen ? 36 : 42,
                                        decoration: BoxDecoration(
                                          shape: BoxShape.circle,
                                          color: const Color(0xA0FFFFFF),
                                        ),
                                        child: Center(
                                          child: SvgPicture.asset(
                                            height: isSmallScreen ? 20 : 25,
                                            width: isSmallScreen ? 20 : 25,
                                            "assets/svg/hrm/location_ic.svg",
                                            fit: BoxFit.contain,
                                          ),
                                        ),
                                      ),
                                    ),
                                  ],
                                ),
                              ),
                            ),
                            SizedBox(width: isSmallScreen ? 8 : 12),
                            Expanded(
                              child: Container(
                                height: getResponsiveHeight(context, 110),
                                padding: EdgeInsets.all(isSmallScreen ? 12 : 16),
                                decoration: BoxDecoration(
                                  color: const Color(0xfffffbc3),
                                  borderRadius: BorderRadius.circular(16),
                                ),
                                child: Stack(
                                  children: [
                                    Column(
                                      crossAxisAlignment:
                                      CrossAxisAlignment.start,
                                      children: [
                                        Text(
                                          "₹$balance", // Balance Amount
                                          style: TextStyle(
                                            fontSize: getResponsiveTextSize(context, 18),
                                            color: const Color(0xff605C00),
                                          ),
                                        ),
                                        SizedBox(height: isSmallScreen ? 6 : 8),
                                        Text(
                                          "Balance \nAmount",
                                          style: TextStyle(
                                            fontSize: getResponsiveTextSize(context, 14),
                                            color: const Color(0xff2D2D2D),
                                            fontWeight: FontWeight.w400,
                                          ),
                                        ),
                                      ],
                                    ),
                                    Positioned(
                                      bottom: 2,
                                      right: 2,
                                      child: Container(
                                        height: isSmallScreen ? 36 : 42,
                                        width: isSmallScreen ? 36 : 42,
                                        decoration: BoxDecoration(
                                          shape: BoxShape.circle,
                                          color: const Color(0xA0FFFFFF),
                                        ),
                                        child: Center(
                                          child: SvgPicture.asset(
                                            height: isSmallScreen ? 20 : 25,
                                            width: isSmallScreen ? 20 : 25,
                                            "assets/svg/hrm/ballance_ic.svg",
                                            fit: BoxFit.contain,
                                          ),
                                        ),
                                      ),
                                    ),
                                  ],
                                ),
                              ),
                            ),
                          ],
                        ),

                        SizedBox(height: getResponsiveHeight(context, 20)),

                        /// --- Reward List Card ---
                        if (rewards != null)
                          _rewardListCard(
                            context: context,
                            title: rewards.description ?? "-",
                            dateTime: rewards.dateTime ?? "-",
                            achieved: achieved,
                            disbursed: disbursed,
                            balance: balance,
                            enteredBy: rewards.enteredBy ?? "-",
                          )
                        else
                          Text(
                            "No rewards available",
                            style: TextStyle(
                              fontSize: getResponsiveTextSize(context, 14),
                            ),
                          ),
                      ],
                    ),
                  );
                },
              ),
            );
          },
        ),
      ),
    );
  }

  ///  Reusable Reward Card Function
  Widget _rewardListCard({
    required BuildContext context,
    required String title,
    required String dateTime,
    required String achieved,
    required String disbursed,
    required String balance,
    required String enteredBy,
  }) {
    final bool isSmallScreen = MediaQuery.of(context).size.width < 360;

    return Container(
      margin: EdgeInsets.only(bottom: getResponsiveHeight(context, 16)),
      padding: EdgeInsets.all(isSmallScreen ? 12 : 16),
      decoration: BoxDecoration(
        color: Colors.white,
        borderRadius: BorderRadius.circular(16),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          /// Header Row
          Row(
            children: [
              CircleAvatar(
                radius: isSmallScreen ? 18 : 22.5,
                backgroundColor: Color(0xffEDF8FF),
                child: SvgPicture.asset(
                  height: isSmallScreen ? 22 : 28,
                  width: isSmallScreen ? 22 : 28,
                  "assets/svg/hrm/rewardList.svg",
                  fit: BoxFit.contain,
                ),
              ),
              SizedBox(width: isSmallScreen ? 6 : 8),
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      title,
                      style: TextStyle(
                        fontSize: getResponsiveTextSize(context, 14.5),
                        color: const Color(0xff2D2D2D),
                        fontFamily: "Plus Jakarta Sans",
                        fontStyle: FontStyle.normal,
                        fontWeight: FontWeight.w400,
                      ),
                      maxLines: 2,
                      overflow: TextOverflow.ellipsis,
                    ),
                    SizedBox(height: isSmallScreen ? 2 : 4),
                    Text(
                      dateTime,
                      style: TextStyle(
                        fontSize: getResponsiveTextSize(context, 12.5),
                        color: const Color(0xff818181),
                        fontFamily: "Plus Jakarta Sans",
                        fontStyle: FontStyle.normal,
                        fontWeight: FontWeight.w400,
                      ),
                    ),
                  ],
                ),
              ),
            ],
          ),

          SizedBox(height: isSmallScreen ? 8 : 12),

          /// Amount Details
          Padding(
            padding: const EdgeInsets.all(2.0),
            child: Row(
              children: [
                Text(
                  "Amount Details",
                  style: TextStyle(
                      fontSize: getResponsiveTextSize(context, 14),
                      fontFamily: "JakartaSemiBold"
                  ),
                ),
                SizedBox(width: isSmallScreen ? 6 : 10),
                Expanded(
                  child: DottedLine(
                    dashGapLength: 4,
                    dashGapColor: Colors.white,
                    dashColor: AppColors.grey_semi,
                    dashLength: 2,
                    lineThickness: 0.5,
                  ),
                ),
              ],
            ),
          ),
          SizedBox(height: isSmallScreen ? 4 : 6),

          _buildKeyValue(context, "Achieved Amount", achieved),
          _buildKeyValue(context, "Disbursed Amount", disbursed),
          _buildKeyValue(context, "Balance Amount", balance),

          SizedBox(height: isSmallScreen ? 8 : 10),

          /// Employee Details
          Padding(
            padding: const EdgeInsets.all(2.0),
            child: Row(
              children: [
                Text(
                  "Employee Details",
                  style: TextStyle(
                      fontSize: getResponsiveTextSize(context, 14),
                      fontFamily: "JakartaSemiBold"
                  ),
                ),
                SizedBox(width: isSmallScreen ? 6 : 10),
                Expanded(
                  child: DottedLine(
                    dashGapLength: 4,
                    dashGapColor: Colors.white,
                    dashColor: AppColors.grey_semi,
                    dashLength: 2,
                    lineThickness: 0.5,
                  ),
                ),
              ],
            ),
          ),

          _buildKeyValue(context, "Entered By", enteredBy),
        ],
      ),
    );
  }

  ///  Key-Value Row
  Widget _buildKeyValue(BuildContext context, String key, String value) {
    final bool isSmallScreen = MediaQuery.of(context).size.width < 360;

    return Padding(
      padding: EdgeInsets.symmetric(
          vertical: isSmallScreen ? 2.5 : 3.5,
          horizontal: 2
      ),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Expanded(
            flex: 2,
            child: Text(
              key,
              style: TextStyle(
                fontFamily: "JakartaRegular",
                fontSize: getResponsiveTextSize(context, 14),
                color: AppColors.semi_black,
              ),
              maxLines: 1,
              overflow: TextOverflow.ellipsis,
            ),
          ),
          Expanded(
            flex: 1,
            child: Text(
              value,
              style: TextStyle(
                  fontSize: getResponsiveTextSize(context, 14),
                  color: const Color(0xFF818181)
              ),
              textAlign: TextAlign.right,
              maxLines: 1,
              overflow: TextOverflow.ellipsis,
            ),
          ),
        ],
      ),
    );
  }
}